import tkinter as tk
from tkinter import ttk, filedialog, scrolledtext, messagebox
from PIL import Image, ImageTk
import threading
import time
import os

class IntentionRepeaterGUI(tk.Tk):
    def __init__(self):
        super().__init__()

        self.title("Intention Repeater GUI Basic")
        self.geometry("800x850")

        # Center the window on the screen
        self.update_idletasks()
        width = self.winfo_width()
        height = self.winfo_height()
        x = (self.winfo_screenwidth() // 2) - (width // 2)
        y = (self.winfo_screenheight() // 2) - (height // 2)
        self.geometry(f'{width}x{height}+{x}+{y}')

        self.running = False
        self.iterations = 0
        self.start_time = 0

        # Paths for images, initialized to None
        self.subject_image_path = None
        self.witness_image_path = None

        self.create_widgets()

    def create_widgets(self):
        main_frame = ttk.Frame(self, padding="10")
        main_frame.pack(fill=tk.BOTH, expand=True)

        # Intention Frame
        intention_frame = ttk.LabelFrame(main_frame, text="Step 1: Define the Intention")
        intention_frame.pack(fill=tk.X, pady=5)
        
        description_label = ttk.Label(intention_frame, text="This is the core message or energy to be repeated.", font=("Helvetica", 9, "italic"))
        description_label.pack(anchor=tk.W, padx=5, pady=(2, 5))

        self.intention_source = tk.StringVar(value="text")

        ttk.Radiobutton(intention_frame, text="Intention Textbox", variable=self.intention_source, value="text").pack(anchor=tk.W, padx=5)
        self.intention_text = scrolledtext.ScrolledText(intention_frame, wrap=tk.WORD, width=60, height=10)
        self.intention_text.pack(fill=tk.BOTH, expand=True, padx=5, pady=5)

        ttk.Radiobutton(intention_frame, text="File", variable=self.intention_source, value="file").pack(anchor=tk.W, padx=5)
        file_frame = ttk.Frame(intention_frame)
        file_frame.pack(fill=tk.X, padx=5, pady=5)
        self.file_path_entry = ttk.Entry(file_frame, width=50)
        self.file_path_entry.pack(side=tk.LEFT, fill=tk.X, expand=True)
        ttk.Button(file_frame, text="Load File", command=self.load_file).pack(side=tk.LEFT, padx=5)

        # Burst Count Frame
        burst_frame = ttk.LabelFrame(main_frame, text="Burst Count")
        burst_frame.pack(fill=tk.X, pady=5)
        self.burst_count = tk.StringVar(value="1")
        ttk.Entry(burst_frame, textvariable=self.burst_count, width=10).pack(padx=5, pady=5)

        # Speed Frame
        speed_frame = ttk.LabelFrame(main_frame, text="Repetition Speed")
        speed_frame.pack(fill=tk.X, pady=5)
        self.speed = tk.StringVar(value="max")
        ttk.Radiobutton(speed_frame, text="Maximum Speed", variable=self.speed, value="max").pack(anchor=tk.W, padx=5)
        ttk.Radiobutton(speed_frame, text="Once-Per-Hour", variable=self.speed, value="hour").pack(anchor=tk.W, padx=5)

        # Subject and Witness Frame
        subject_witness_frame = ttk.Frame(main_frame)
        subject_witness_frame.pack(fill=tk.X, pady=5)

        # Subject Frame
        subject_frame = ttk.LabelFrame(subject_witness_frame, text="Subject (Optional Target)")
        subject_frame.pack(side=tk.LEFT, fill=tk.BOTH, expand=True, padx=(0, 5))
        self.subject_type = tk.StringVar(value="text")
        ttk.Radiobutton(subject_frame, text="Text", variable=self.subject_type, value="text").pack(anchor=tk.W, padx=5)
        self.subject_text = ttk.Entry(subject_frame, width=30)
        self.subject_text.pack(fill=tk.X, padx=5, pady=5)
        ttk.Radiobutton(subject_frame, text="Image", variable=self.subject_type, value="image").pack(anchor=tk.W, padx=5)
        self.subject_image_label = ttk.Label(subject_frame, text="No image loaded", wraplength=150)
        self.subject_image_label.pack(fill=tk.X, padx=5, pady=5)
        ttk.Button(subject_frame, text="Load Image", command=lambda: self.load_image('subject')).pack(pady=5)

        # Witness Frame
        witness_frame = ttk.LabelFrame(subject_witness_frame, text="Witness (Optional Link)")
        witness_frame.pack(side=tk.LEFT, fill=tk.BOTH, expand=True, padx=(5, 0))
        self.witness_type = tk.StringVar(value="text")
        ttk.Radiobutton(witness_frame, text="Text", variable=self.witness_type, value="text").pack(anchor=tk.W, padx=5)
        self.witness_text = ttk.Entry(witness_frame, width=30)
        self.witness_text.pack(fill=tk.X, padx=5, pady=5)
        ttk.Radiobutton(witness_frame, text="Image", variable=self.witness_type, value="image").pack(anchor=tk.W, padx=5)
        self.witness_image_label = ttk.Label(witness_frame, text="No image loaded", wraplength=150)
        self.witness_image_label.pack(fill=tk.X, padx=5, pady=5)
        ttk.Button(witness_frame, text="Load Image", command=lambda: self.load_image('witness')).pack(pady=5)

        # Status Frame
        status_frame = ttk.LabelFrame(main_frame, text="Status")
        # --- MODIFIED LINE HERE ---
        status_frame.pack(fill=tk.X, pady=5, ipady=5)
        # --- END MODIFIED LINE ---
        self.elapsed_time_label = ttk.Label(status_frame, text="Elapsed Time: 00:00:00", font=("Helvetica", 24, "bold"))
        self.elapsed_time_label.pack(pady=10)
        self.iterations_label = ttk.Label(status_frame, text="Iterations: 0", font=("Helvetica", 18))
        self.iterations_label.pack(pady=5)
        self.status_label = ttk.Label(status_frame, text="Stopped", font=("Helvetica", 12))
        self.status_label.pack(pady=5)

        # Control Frame
        control_frame = ttk.Frame(main_frame)
        control_frame.pack(fill=tk.X, pady=10)
        self.start_button = ttk.Button(control_frame, text="Start", command=self.start_repeater)
        self.start_button.pack(side=tk.LEFT, fill=tk.X, expand=True, padx=(0, 5))
        self.stop_button = ttk.Button(control_frame, text="Stop", command=self.stop_repeater, state=tk.DISABLED)
        self.stop_button.pack(side=tk.LEFT, fill=tk.X, expand=True, padx=(5, 0))

    def load_file(self):
        filepath = filedialog.askopenfilename(
            title="Open Intention File",
            filetypes=(("Text files", "*.txt"), ("All files", "*.*"))
        )
        if filepath:
            self.file_path_entry.delete(0, tk.END)
            self.file_path_entry.insert(0, filepath)
            self.intention_source.set("file")

    def load_image(self, image_type):
        filepath = filedialog.askopenfilename(
            title=f"Load {image_type.capitalize()} Image",
            filetypes=(("Image files", "*.png *.jpg *.jpeg *.gif *.bmp"),)
        )
        if filepath:
            if image_type == 'subject':
                self.subject_image_path = filepath
                self.subject_image_label.config(text=os.path.basename(filepath))
                self.subject_type.set("image")
            elif image_type == 'witness':
                self.witness_image_path = filepath
                self.witness_image_label.config(text=os.path.basename(filepath))
                self.witness_type.set("image")

    def start_repeater(self):
        self.running = True
        self.iterations = 0
        self.start_time = time.time()
        self.start_button.config(state=tk.DISABLED)
        self.stop_button.config(state=tk.NORMAL)
        self.status_label.config(text="Running...")

        self.repeater_thread = threading.Thread(target=self.repeater_loop, daemon=True)
        self.repeater_thread.start()
        self.update_status()

    def stop_repeater(self):
        self.running = False
        self.start_button.config(state=tk.NORMAL)
        self.stop_button.config(state=tk.DISABLED)
        self.status_label.config(text="Stopped")

    def repeater_loop(self):
        while self.running:
            try:
                burst_count = int(self.burst_count.get())
            except ValueError:
                burst_count = 1

            intention = ""
            if self.intention_source.get() == "text":
                intention = self.intention_text.get("1.0", tk.END).strip()
            elif self.intention_source.get() == "file":
                try:
                    with open(self.file_path_entry.get(), 'r', encoding='utf-8') as f:
                        intention = f.read()
                except Exception as e:
                    self.after(0, lambda: messagebox.showerror("File Error", f"Could not read file: {e}"))
                    self.after(0, self.stop_repeater)
                    return

            if not intention:
                self.after(0, lambda: messagebox.showwarning("Input Error", "The Intention source is empty."))
                self.after(0, self.stop_repeater)
                return

            for _ in range(burst_count):
                if not self.running:
                    break
                # This is the core "application" of the intention.
                # The 'intention' variable is now charged and conceptually
                # applied to the Subject via the Witness link.
                _ = intention
                self.iterations += 1

            if self.speed.get() == "hour" and self.running:
                self.after(0, lambda: self.status_label.config(text="Sleeping for one hour..."))
                time.sleep(3600)
                if self.running:
                    self.after(0, lambda: self.status_label.config(text="Running..."))


    def update_status(self):
        if self.running:
            elapsed_seconds = int(time.time() - self.start_time)
            hours, remainder = divmod(elapsed_seconds, 3600)
            minutes, seconds = divmod(remainder, 60)
            self.elapsed_time_label.config(text=f"Elapsed Time: {hours:02}:{minutes:02}:{seconds:02}")
            self.iterations_label.config(text=f"Iterations: {self.format_iterations(self.iterations)}")
            self.after(1000, self.update_status)

    def format_iterations(self, n):
        if n < 1000:
            return str(n)
        for unit in ['', 'k', 'M', 'B', 'T', 'Q']:
            if abs(n) < 1000.0:
                return f"{n:3.1f}{unit}"
            n /= 1000.0
        return f"{n:.1f}Q"

if __name__ == "__main__":
    app = IntentionRepeaterGUI()
    app.mainloop()